﻿using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Security.AccessControl;
using System.Security.Principal;
using System.Text;

namespace DuplicateFinder
{
    class Program
    {
        static void Main(string[] args)
        {
            bool recurseIntoSubdirectories = false;

            if (args.Length < 1)
            {
                ShowUsage();
                return;
            }

            int firstDirectoryIndex = 0;
            IEnumerable<string> directoriesToSearch = null;
            bool testDirectoriesMade = false;

            // Sprawdzamy, czy program działa w trybie testowym.
            if (args.Length == 1 && args[0] == "/test")
            {
                directoriesToSearch = MakeTestDirectories();
                testDirectoriesMade = true;
                recurseIntoSubdirectories = true;
            }
            else
            {
                if (args.Length > 1)
                {
                    // Sprawdzamy, czy katalogi mają być sprawdzane rekurencyjnie.
                    if (args[0] == "/sub")
                    {
                        if (args.Length < 2)
                        {
                            ShowUsage();
                            return;
                        }
                        recurseIntoSubdirectories = true;
                        firstDirectoryIndex = 1;
                    }
                }

                // Pobranie listy katalogów z wiersza poleceń.
                directoriesToSearch = args.Skip(firstDirectoryIndex);
            }

            try
            {
                List<FileNameGroup> filesGroupedByName =
                    InspectDirectories(recurseIntoSubdirectories, directoriesToSearch);

                DisplayMatches(filesGroupedByName);
                Console.ReadKey();
            }
            catch (PathTooLongException ptlx)
            {
                Console.WriteLine("Podana ścieżka była zbyt długa.");
                Console.WriteLine(ptlx.Message);
            }
            catch (DirectoryNotFoundException dnfx)
            {
                Console.WriteLine("Nie znaleziono podanego katalogu.");
                Console.WriteLine(dnfx.Message);
            }
            catch (IOException iox)
            {
                Console.WriteLine(iox.Message);
            }
            catch (UnauthorizedAccessException uax)
            {
                Console.WriteLine("Brak uprawnień do dostępu do podanego katalogu.");
                Console.WriteLine(uax.Message);
            }
            catch (ArgumentException ax)
            {
                Console.WriteLine("Podana ścieżka była nieprawidłowa.");
                Console.WriteLine(ax.Message);
            }
            finally
            {
                if (testDirectoriesMade)
                {
                    CleanupTestDirectories(directoriesToSearch);
                }
            }
        }

        private static void ShowUsage()
        {
            Console.WriteLine("Odnajdywanie duplikatów plików.");
            Console.WriteLine("====================");
            Console.WriteLine(
                "Program poszukuje powtarzających się plików w jednym lub kilku katalogach.");
            Console.WriteLine();
            Console.WriteLine(
                "Sposób korzystania: findduplicatefiles [/sub] NazwaKatalogu [NazwaKatalogu] ...");
            Console.WriteLine("/sub - rekurencyjne przeszukiwanie podkatalogów.");
            Console.ReadKey();
        }

        private static List<FileNameGroup> InspectDirectories(
            bool recurseIntoSubdirectories,
            IEnumerable<string> directoriesToSearch)
        {
            var searchOption = recurseIntoSubdirectories ?
                SearchOption.AllDirectories : SearchOption.TopDirectoryOnly;

            // Pobranie ścieżki dostępu do każdego pliku w każdym 
            // z przeszukiwanych katalogów

            var allFilePaths = from directory in directoriesToSearch
                               from file in GetDirectoryFiles(directory,
                                                              searchOption)
                               select file;

            // Grupujemy pliki według nazwy lokalnej (czyli nazwy bez ścieżki
            // dostępu), i dla każdej z tych nazw tworzymy listę, zawierającą
            // szczegółowe informacje o każdym pliku o danej nazwie.

            var fileNameGroups = from filePath in allFilePaths
                                 let fileNameWithoutPath = Path.GetFileName(filePath)
                                 group filePath by fileNameWithoutPath into nameGroup
                                 select new FileNameGroup
                                 {
                                     FileNameWithoutPath = nameGroup.Key,
                                     FilesWithThisName = GetDetails(nameGroup).ToList()
                                 };

            return fileNameGroups.ToList();
        }

        // Listing 11-33. Zmodyfikowana metoda DisplayMatches porównująca zawartości plików.
        private static void DisplayMatches(
            IEnumerable<FileNameGroup> filesGroupedByName)
        {
            var groupsWithMoreThanOneFile = from nameGroup in filesGroupedByName
                                            where nameGroup.FilesWithThisName.Count > 1
                                            select nameGroup;

            foreach (var fileNameGroup in groupsWithMoreThanOneFile)
            {
                // Pogrupowanie odpowiadających sobie plików na podstawie ich wielkości
                // i wybranie tych, których dla danej wielkości jest większa od  jeden.
                var matchesBySize = from file in fileNameGroup.FilesWithThisName
                                    group file by file.FileSize into sizeGroup
                                    where sizeGroup.Count() > 1
                                    select sizeGroup;

                foreach (var matchedBySize in matchesBySize)
                {
                    string fileNameAndSize = string.Format("{0} ({1} bajtów)",
                    fileNameGroup.FileNameWithoutPath, matchedBySize.Key);
                    WriteWithUnderlines(fileNameAndSize);

                    // Wyświetlenie każdego katalogu zawierającego dany plik.
                    {
                        List<FileContents> content = LoadFiles(matchedBySize);
                        CompareFiles(content);
                    }
                    Console.WriteLine();
                }
            }
        }

        // Listing 11-35. Wczytywanie binarnej zawartości pliku.
        private static List<FileContents> LoadFiles(IEnumerable<FileDetails> fileList)
        {
            var content = new List<FileContents>();
            foreach (FileDetails item in fileList)
            {
                byte[] contents = File.ReadAllBytes(item.FilePath);
                content.Add(new FileContents
                {
                    FilePath = item.FilePath,
                    Content = contents
                });
            }
            return content;
        }

        // Listing 11-36. Metoda CompareFiles.
        private static void CompareFiles(List<FileContents> files)
        {
            Dictionary<FileContents, List<FileContents>> potentiallyMatched =
            BuildPotentialMatches(files);

            // Teraz porównamy poszczególne bajty obu plików.
            CompareBytes(files, potentiallyMatched);

            DisplayResults(files, potentiallyMatched);
        }

        // Listing 11-37. Tworzenie kombinacji potencjalnych duplikatów.
        private static Dictionary<FileContents, List<FileContents>>
            BuildPotentialMatches(List<FileContents> files)
        {
            // Metoda tworzy słownik, którego elementy mają następującą postać:
            // { 0, { 1, 2, 3, 4, ... N } }
            // { 1, { 2, 3, 4, ... N }
            // ...
            // { N - 1, { N } }
            // gdzie N jest równe liczbie plików pomniejszonej o jeden.
            var allCombinations = Enumerable.Range(0, files.Count - 1).ToDictionary(
                x => files[x],
            x => files.Skip(x + 1).ToList());

            return allCombinations;
        }

        // Listing 11-38. Wyświetlanie duplikatów
        private static void DisplayResults(
            List<FileContents> files,
            Dictionary<FileContents, List<FileContents>> currentlyMatched)
        {
            if (currentlyMatched.Count == 0) { return; }

            var alreadyMatched = new List<FileContents>();

            Console.WriteLine("Duplikaty");

            foreach (var matched in currentlyMatched)
            {
                // Nie sprawdzamy, jeśli plik został już wcześniej dopasowany.
                if (alreadyMatched.Contains(matched.Key))
                {
                    continue;
                }
                else
                {
                    alreadyMatched.Add(matched.Key);
                }
                Console.WriteLine("-------");
                Console.WriteLine(matched.Key.FilePath);
                foreach (var file in matched.Value)
                {
                    Console.WriteLine(file.FilePath);
                    alreadyMatched.Add(file);
                }
            }
            Console.WriteLine("-------");
        }

        // Listing 11-39. Porównywanie zawartości potencjalnych duplikatów bajt po bajcie.
        private static void CompareBytes(
            List<FileContents> files,
            Dictionary<FileContents, List<FileContents>> potentiallyMatched)
        {
            // Metoda jest wywoływana wyłącznie wtedy, gdy pliki mają identyczną długość.
            int fileLength = files[0].Content.Length;
            var sourceFilesWithNoMatches = new List<FileContents>();
            for (int fileByteOffset = 0; fileByteOffset < fileLength; ++fileByteOffset)
            {
                foreach (var sourceFileEntry in potentiallyMatched)
                {
                    byte[] sourceContent = sourceFileEntry.Key.Content;
                    for (int otherIndex = 0; otherIndex < sourceFileEntry.Value.Count;
                    ++otherIndex)
                    {
                        // Sprawdzamy bajty o określonym indeksie w obu plikach. Jeśli różnią 
                        // się one od siebie, to usuwamy pliki z kolekcji.
                        byte[] otherContent =
                        sourceFileEntry.Value[otherIndex].Content;
                        if (sourceContent[fileByteOffset] != otherContent[fileByteOffset])
                        {
                            sourceFileEntry.Value.RemoveAt(otherIndex);
                            otherIndex -= 1;
                            if (sourceFileEntry.Value.Count == 0)
                            {
                                sourceFilesWithNoMatches.Add(sourceFileEntry.Key);
                            }
                        }
                    }
                }
                foreach (FileContents fileWithNoMatches in sourceFilesWithNoMatches)
                {
                    potentiallyMatched.Remove(fileWithNoMatches);
                }
                // Nie zawracamy sobie głowy resztą plików, jeśli
                // nie ma już żadnych innych potencjalnych duplikatów.
                if (potentiallyMatched.Count == 0)
                {
                    break;
                }
                sourceFilesWithNoMatches.Clear();
            }
        }

        private static void WriteWithUnderlines(string text)
        {
            Console.WriteLine(text);
            Console.WriteLine(new string('-', text.Length));
        }

        private static string[] MakeTestDirectories()
        {
            string localApplicationData = Path.Combine(
                Environment.GetFolderPath(
                    Environment.SpecialFolder.LocalApplicationData),
                @"Programming CSharp\FindDuplicates");

            // Pobranie nazwy bieżącego użytkownika
            string userName = WindowsIdentity.GetCurrent().Name;
            // Określenie reguły kontroli dostępu
            FileSystemAccessRule fsarAllow =
                new FileSystemAccessRule(
                    userName,
                    FileSystemRights.FullControl,
                    AccessControlType.Allow);
            DirectorySecurity ds = new DirectorySecurity();
            ds.AddAccessRule(fsarAllow);

            FileSystemAccessRule fsarDeny =
                new FileSystemAccessRule(
                    userName,
                    FileSystemRights.WriteExtendedAttributes,
                    AccessControlType.Deny);
            ds.AddAccessRule(fsarDeny);

            // Tworzymy trzy katalogi testowe
            // i zostawiamy miejsce na czwarty, by przetestować działanie programu
            // w przypadku braku praw dostępu do katalogu.
            var directories = new string[4];
            for (int i = 0; i < directories.Length - 1; ++i)
            {
                string directory = Path.GetRandomFileName();
                // Łączymy dane lokalnej aplikacji z losowymi 
                // nazwami plików i katalogów
                string fullPath = Path.Combine(localApplicationData, directory);
                // I tworzymy katalog.
                Directory.CreateDirectory(fullPath, ds);
                directories[i] = fullPath;
                Console.WriteLine(fullPath);
            }

            CreateTestFiles(directories.Take(3));

            directories[3] = CreateDeniedDirectory(localApplicationData);

            return directories;
        }

        private static void CleanupTestDirectories(IEnumerable<string> directories)
        {
            foreach (var directory in directories)
            {
                AllowAccess(directory);
                Directory.Delete(directory, true);
            }
        }

        private static void CreateTestFiles(IEnumerable<string> directories)
        {
            string fileForAllDirectories = "SameNameAndContent.txt";
            string fileSameInAllButDifferentSizes = "SameNameDifferentSize.txt";
            string fileSameSizeInAllButDifferentContent =
                "SameNameAndSizeDifferentContent.txt";

            int directoryIndex = 0;
            // Tworzymy unikalny plik dla danego katalogu.
            foreach (string directory in directories)
            {
                directoryIndex++;

                // Tworzymy unikalny plik dla tego katalogu.
                string filename = Path.GetRandomFileName();
                string fullPath = Path.Combine(directory, filename);
                CreateFile(fullPath, "Przykładowa zawartość nr 1");

                // A teraz plik, który będzie się pojawiał we wszystkich katalogach
                // i będzie miał tę samą zawartość.
                fullPath = Path.Combine(directory, fileForAllDirectories);
                CreateFile(fullPath, "Jestem w każdym katalogu");

                // I kolejny plik, który w każdym katalogu będzie miał taką samą
                // nazwę, lecz inną zawartość.
                fullPath = Path.Combine(directory, fileSameInAllButDifferentSizes);
                StringBuilder builder = new StringBuilder();
                builder.AppendLine("Nowa zawartość: ");
                builder.AppendLine(new string('x', directoryIndex));
                CreateFile(fullPath, builder.ToString());

                // A teraz plik o tej samej długości, lecz innej zawartości.
                fullPath = Path.Combine(directory, fileSameSizeInAllButDifferentContent);
                builder = new StringBuilder();
                builder.Append("Nowa zawartość i dodatkowo liczba ");
                builder.Append(directoryIndex);
                builder.AppendLine(".");
                CreateFile(fullPath, builder.ToString());
            }
        }

        private static void CreateFile(string fullPath, string contents)
        {
            File.WriteAllText(fullPath, contents);
        }

        static void DoubtfulCode()
        {
            if (File.Exists("SomeFile.txt"))
            {
                // Tu wykonujemy operacje na pliku.
            }
        }

        private static string CreateDeniedDirectory(string parentPath)
        {
            string deniedDirectory = Path.GetRandomFileName();
            string fullDeniedPath = Path.Combine(parentPath, deniedDirectory);
            string userName = WindowsIdentity.GetCurrent().Name;
            DirectorySecurity ds = new DirectorySecurity();
            FileSystemAccessRule fsarDeny =
                new FileSystemAccessRule(
                    userName,
                    FileSystemRights.ListDirectory,
                    AccessControlType.Deny);
            ds.AddAccessRule(fsarDeny);

            Directory.CreateDirectory(fullDeniedPath, ds);
            return fullDeniedPath;
        }

        private static void AllowAccess(string directory)
        {
            DirectorySecurity ds = Directory.GetAccessControl(directory);

            string userName = WindowsIdentity.GetCurrent().Name;

            // Usuwamy regułę odbierającą uprawnienia...
            FileSystemAccessRule fsarDeny =
                new FileSystemAccessRule(
                    userName,
                    FileSystemRights.ListDirectory,
                    AccessControlType.Deny);
            ds.RemoveAccessRuleSpecific(fsarDeny);

            // ...i dodajemy regułę przyznającą wszystkie uprawnienia.
            FileSystemAccessRule fsarAllow =
                new FileSystemAccessRule(
                    userName,
                    FileSystemRights.FullControl,
                    AccessControlType.Allow);
            ds.AddAccessRule(fsarAllow);

            Directory.SetAccessControl(directory, ds);
        }

        private static IEnumerable<string> GetDirectoryFiles(
            string directory, SearchOption searchOption)
        {
            try
            {
                return Directory.GetFiles(directory, "*.*", searchOption);
            }
            catch (DirectoryNotFoundException dnfx)
            {
                Console.WriteLine("Ostrzeżenie: Nie znaleziono podanego katalogu.");
                Console.WriteLine(dnfx.Message);
            }
            catch (UnauthorizedAccessException uax)
            {
                Console.WriteLine(
                "Ostrzeżenie: Brak uprawnień do dostępu do wskazanego katalogu.");
                Console.WriteLine(uax.Message);
            }
            return Enumerable.Empty<string>();
        }

        private static IEnumerable<FileDetails> GetDetails(IEnumerable<string> paths)
        {
            foreach (string filePath in paths)
            {
                FileDetails details = null;
                try
                {
                    FileInfo info = new FileInfo(filePath);
                    details = new FileDetails
                    {
                        FilePath = filePath,
                        FileSize = info.Length
                    };
                }
                catch (FileNotFoundException fnfx)
                {
                    Console.WriteLine("Ostrzeżenie: Nie znaleziono podanego pliku.");
                    Console.WriteLine(fnfx.Message);
                }
                catch (IOException iox)
                {
                    Console.Write("Ostrzeżenie: ");
                    Console.WriteLine(iox.Message);
                }
                catch (UnauthorizedAccessException uax)
                {
                    Console.WriteLine(
                    "Ostrzeżenie: Brak uprawnień do dostępu do podanego pliku.");
                    Console.WriteLine(uax.Message);
                }
                if (details != null)
                {
                    yield return details;
                }
            }
        }
    }
}
